<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package   auth_wdmwpmoodle
 * @copyright (c) 2021 WisdmLabs (https://wisdmlabs.com/) <support@wisdmlabs.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @author    Ishwar Singh
 */

/**
 * Serves the files from the auth_wdmwpmoodle file areas
 *
 * @param stdClass $course the course object
 * @param stdClass $cm the course module object
 * @param stdClass $context the auth_wdmwpmoodle's context
 * @param string   $filearea the name of the file area
 * @param array    $args extra arguments (itemid, path)
 * @param bool     $forcedownload whether or not force download
 * @param array    $options additional options affecting the file serving
 */
function auth_wdmwpmoodle_pluginfile(
	$course,
	$cm,
	$context,
	$filearea,
	array $args,
	$forcedownload = 0,
	array $options = array()
) {
	if ( $context->contextlevel != CONTEXT_SYSTEM ) {
		send_file_not_found();
	}
	$itemid       = (int) array_shift( $args );
	$relativepath = implode( '/', $args );
	$fullpath     = "/{$context->id}/auth_wdmwpmoodle/$filearea/$itemid/$relativepath";
	$fs           = get_file_storage();
	if ( ! ( $file = $fs->get_file_by_hash( sha1( $fullpath ) ) ) ) {
		return false;
	}
	// Download MUST be forced - security!
	send_stored_file( $file, 0, 0, $forcedownload, $options );
}

/**
 * Check active webservices and update functions for auth_wdmwpmoodle plugin.
 */
function check_and_update_webservice_functions_wdmwpmoodle() {

	global $DB, $CFG;
	// get connection settings.
	$connections = isset($CFG->eb_connection_settings) ? unserialize($CFG->eb_connection_settings) : array();
	
	foreach ( $connections as $connection ) {
		$data = $DB->get_record('external_tokens', array('token' => $connection['wp_token']), 'externalserviceid');
		$serviceid = isset($data->externalserviceid) ? $data->externalserviceid : '';

		if( empty($serviceid) ) {
			continue;
		}

		$ssofunctions = array(
            array('externalserviceid' => $serviceid, 'functionname' => 'wdm_sso_verify_token'),
        );

		foreach ( $ssofunctions as $function ) {

			// adding function without check because services.php runs after install.php 
			// and at this time there are no functions from this plugin.
			$DB->insert_record( 'external_services_functions', $function );
		}
	}
}

/**
 * Enable the plugin in the default authentication method.
 */
function enable_wdmwpmoodle_plugin() {
	global $DB, $CFG;

	$auth = 'wdmwpmoodle';
	get_enabled_auth_plugins(true); // fix the list of enabled auths
	if (empty($CFG->auth)) {
		$authsenabled = array();
	} else {
		$authsenabled = explode(',', $CFG->auth);
	}
	if (!empty($auth) and !exists_auth_plugin($auth)) {
		return false;
	}
	if (!in_array($auth, $authsenabled)) {
		$authsenabled[] = $auth;
		$authsenabled = array_unique($authsenabled);
		set_config('auth', implode(',', $authsenabled));
	}
	\core\session\manager::gc(); // Remove stale sessions.
	core_plugin_manager::reset_caches();
}
